/*
  fft2d_host.ldf

  Copyright (C) 2012 Adapteva, Inc.
  Contributed by Yainv Sapir <yaniv@adapteva.com>

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program, see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>.
*/


// This program is the host part of the fft2d() example project.
//
// This program runs on the linux host and invokes the Epiphany fft2d()
// implementation. It communicates with the system via the eHost library.
// After establishing a connection using the e-server, input image is
// read and parsed using the DevIL library API. A reference
// calculation is done on the host and is compared to the Epiphany
// result. A succes/error message is printed on the terminal according
// to the result of the comparison.
//
// May-2012, YS.

#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <sys/time.h>

#include <IL/il.h>
#define ILU_ENABLED
#define ILUT_ENABLED
/* We would need ILU just because of iluErrorString() function... */
/* So make it possible for both with and without ILU!  */
#ifdef ILU_ENABLED
#include <IL/ilu.h>
#define PRINT_ERROR_MACRO printf("IL Error: %s\n", iluErrorString(Error))
#else /* not ILU_ENABLED */
#define PRINT_ERROR_MACRO printf("IL Error: 0x%X\n", (unsigned int) Error)
#endif /* not ILU_ENABLED */
#ifdef ILUT_ENABLED
#include <IL/ilut.h>
#endif

typedef unsigned int e_coreid_t;
#include <e_host.h>
#include "fft2dlib.h"
#include "fft2d.h"
#include "dram_buffers.h"

#define eMHz 400e6
#define EPS (0.0002)
#define RdBlkSz 1024

int  main(int argc, char *argv[]);
void matrix_init(int seed);
int  fft2d_go();
void init_coreID(unsigned int *coreID, int rows, int cols, unsigned int base_core);

cfloat Bref[_Smtx];
cfloat Bdiff[_Smtx];

unsigned int DRAM_BASE  = 0x81000000;
unsigned int BankA_addr = 0x2000;
unsigned int coreID[_Ncores];

typedef struct timeval timeval_t;


int main(int argc, char *argv[])
{
	const char *servIP = "127.0.0.1";
	const unsigned short eServLoaderPort = 50999;



	ILuint	ImgId;
//	ILenum	Error;
	ILubyte *imdata;
	ILuint  imsize, imBpp;



	int cnum;
	uint32_t time_f, time_s, time_e;
	unsigned int addr;
	size_t sz;
	timeval_t timer[4];
	FILE *fo;
//	FILE *fi;
	char ifname[255], ofname[255], buf[255];
	char *dotp;
	


	// Process input arguments
	if (argc < 2)
	{
		fprintf(stderr, "Usage: fft2d_host <image-file>\n");
		fprintf(stderr, "   The program will open the image file ""image-file"", apply a low-pass\n");
		fprintf(stderr, "   filter and save the filtered image to ""lpf.image-file""\n\n");
		exit(1);
	} else {
		strcpy(ifname, argv[1]);
		strcpy(ofname, ifname);
		dotp = strrchr(ofname, '.');
		if (dotp != NULL)
		{
			strcpy(buf, dotp);
			strcpy(dotp, ".lpf");
			strcat(ofname, buf);
		} else {
			strcat(ofname, ".lpf");
		}
	}

	// Check if the DevIL shared lib's version matches the executable's version.
	if (ilGetInteger(IL_VERSION_NUM) < IL_VERSION)
	{
		fprintf(stderr, "DevIL version is different ...exiting!\n");
		exit(2);
	}

	// Initialize DevIL.
	ilInit();
#ifdef ILU_ENABLED
	iluInit();
#endif



	// create the coreID list
	init_coreID(coreID, _Nside, _Nside, 0x824);


	// Generate the main image name to use, bind it and load the image file.
	ilGenImages(1, &ImgId);
	ilBindImage(ImgId);
	printf("\n");
	printf("Loading original image from file \"%s\".\n\n", ifname);
	if (!ilLoadImage(ifname))
	{
		fprintf(stderr, "Could not open input image file \"%s\" ...exiting.\n", ifname);
		exit(3);
	}

	//	fi = fopen(ifname, "rb");
	fo = stdout;
	fo = fopen("matprt.m", "w");
	if ((fo == NULL)) // || (fi == NULL))
	{
		fprintf(stderr, "Could not open Octave file \"%s\" ...exiting.\n", "matprt.m");
		exit(4);
	}


	// Display the image's dimensions to the end user.
	printf("Width: %d  Height: %d  Depth: %d  Bpp: %d\n\n",
	       ilGetInteger(IL_IMAGE_WIDTH),
	       ilGetInteger(IL_IMAGE_HEIGHT),
	       ilGetInteger(IL_IMAGE_DEPTH),
	       ilGetInteger(IL_IMAGE_BITS_PER_PIXEL));

	imdata = ilGetData();
	imsize = ilGetInteger(IL_IMAGE_WIDTH) * ilGetInteger(IL_IMAGE_HEIGHT);
	imBpp  = ilGetInteger(IL_IMAGE_BYTES_PER_PIXEL);

	if (imsize != (_Sfft * _Sfft))
	{
		printf("Image file size is different from %dx%d ...exiting.\n", _Sfft, _Sfft);
		exit(5);
	}


	// Extract image data into the A matrix.
	for (int i=0; i<imsize; i++)
	{
		Mailbox.A[i] = (float) imdata[i*imBpp] + 0.0 * I;
	}

	fprintf(fo, "\n");

	// Connect to e-server for communicating with the Epiphany system
	if (e_open((char *) servIP, eServLoaderPort))
	{
		fprintf(stderr, "\nERROR: Can't establish connection to E-SERVER!\n\n");
		exit(6);
	}



	// Generate operand matrices based on a provided seed
	matrix_init(0);

#ifdef _USE_DRAM_
	// Copy operand matrices to Epiphany system
	addr = DRAM_BASE + offsetof(shared_buf_t, A[0]);
	sz = sizeof(Mailbox.A);
	 printf(       "Writing A[%ldB] to address %08x...\n", sz, addr);
	fprintf(fo, "%% Writing A[%ldB] to address %08x...\n", sz, addr);
	e_write(addr, (void *) Mailbox.A, sz);

	addr = DRAM_BASE + offsetof(shared_buf_t, B[0]);
	sz = sizeof(Mailbox.B);
	 printf(       "Writing B[%ldB] to address %08x...\n", sz, addr);
	fprintf(fo, "%% Writing B[%ldB] to address %08x...\n", sz, addr);
	e_write(addr, (void *) Mailbox.B, sz);
#else
	// Copy operand matrices to Epiphany cores' memory
	 printf(       "Writing image to Epiphany\n");
	fprintf(fo, "%% Writing image to Epiphany\n");

	sz = sizeof(Mailbox.A) / _Ncores;
	for (cnum=0; cnum<_Ncores; cnum++)
	{
		addr = (coreID[cnum] << 20) | BankA_addr;
		printf(".");
		fflush(stdout);
//		 printf(       "Writing A[%ldB] to address %08x...\n", sz, addr);
		fprintf(fo, "%% Writing A[%ldB] to address %08x...\n", sz, addr);
		e_write(addr, (void *) &Mailbox.A[cnum * _Score * _Sfft], sz);
	}
	printf("\n");
#endif



	// Call the Epiphany fft2d() function
	 printf(       "GO!\n");
	fprintf(fo, "%% GO!\n");
	gettimeofday(&timer[0], NULL);
	fft2d_go();
	gettimeofday(&timer[1], NULL);
	 printf(       "Done!\n");
	fprintf(fo, "%% Done!\n");

	// Read time counters
	addr = DRAM_BASE + offsetof(shared_buf_t, core.time_f);
	sz = sizeof(Mailbox.core.time_f);
//	 printf(       "Reading time count...\n");
	fprintf(fo, "%% Reading time count...\n");
	e_read(addr, (void *) (&time_f), sz);
	addr = DRAM_BASE + offsetof(shared_buf_t, core.time_s);
	e_read(addr, (void *) (&time_s), sz);
	addr = DRAM_BASE + offsetof(shared_buf_t, core.time_e);
	e_read(addr, (void *) (&time_e), sz);

	 printf(       "Finished calculation in %u cycles (%4.2f msec @ %3.0f MHz)\n\n", time_f, (time_f * 1000.0 / eMHz), (eMHz / 1e6));
	fprintf(fo, "%% Finished calculation in %u cycles (%4.2f msec @ %3.0f MHz)\n\n", time_f, (time_f * 1000.0 / eMHz), (eMHz / 1e6));

	 printf(       "Reading processed image back to host\n");
	fprintf(fo, "%% Reading processed image back to host\n");



	// Read result matrix
#ifdef _USE_DRAM_
	addr = DRAM_BASE + offsetof(shared_buf_t, B[0]);
	sz = sizeof(Mailbox.B);
	 printf(       "Reading B[%ldB] from address %08x...\n", sz, addr);
	fprintf(fo, "%% Reading B[%ldB] from address %08x...\n", sz, addr);
	blknum = sz / RdBlkSz;
	remndr = sz % RdBlkSz;
	for (i=0; i<blknum; i++)
	{
		printf(".");
		fflush(stdout);
		e_read(addr+i*RdBlkSz, (void *) ((long unsigned)(Mailbox.B)+i*RdBlkSz), RdBlkSz);
	}
	printf(".");
	fflush(stdout);
	e_read(addr+i*RdBlkSz, (void *) ((long unsigned)(Mailbox.B)+i*RdBlkSz), remndr);
#else
	// Read result matrix from Epiphany cores' memory
	sz = sizeof(Mailbox.A) / _Ncores;
	for (cnum=0; cnum<_Ncores; cnum++)
	{
		addr = (coreID[cnum] << 20) | BankA_addr;
		printf(".");
		fflush(stdout);
//		printf(        "Reading A[%ldB] from address %08x...\n", sz, addr);
		fprintf(fo, "%% Reading A[%ldB] from address %08x...\n", sz, addr);
		e_read(addr, (void *) &Mailbox.B[cnum * _Score * _Sfft], sz);
	}
#endif
	printf("\n");

	fflush(fo);
	fclose(fo);



	// Convert processed image matrix B into the image file date.
	for (int i=0; i<imsize; i++)
	{
		for (int j=0; j<imBpp; j++)
			imdata[i*imBpp+j] = cabs(Mailbox.B[i]);
	}

	// Save processed image to the output file.
	ilEnable(IL_FILE_OVERWRITE);
	printf("\nSaving processed image to file \"%s\".\n\n", ofname);
	if (!ilSaveImage(ofname))
	{
		fprintf(stderr, "Could not open output image file \"%s\" ...exiting.\n", ofname);
		exit(7);
	}

	// We're done with the image, so let's delete it.
	ilDeleteImages(1, &ImgId);

	// Simple Error detection loop that displays the Error to the user in a human-readable form.
//	while ((Error = ilGetError()))
//		PRINT_ERROR_MACRO;

	// Close connection to e-server
	if (e_close())
	{
		fprintf(stderr, "\nERROR: Can't close connection to E-SERVER!\n\n");
		exit(8);
	}

	return 0;
}


// Call (invoke) the fft2d() function
int fft2d_go()
{
	unsigned int addr;
	size_t sz;
	
	// Signal cores to start crunching
	Mailbox.core.go = 1;
	addr = DRAM_BASE + offsetof(shared_buf_t, core.go);
	sz = sizeof(int);
	e_write(addr, (void *) (&Mailbox.core.go), sz);


	// Wait until cores finished calculation
	addr = DRAM_BASE + offsetof(shared_buf_t, core.ready);
	sz = sizeof(int);
	Mailbox.core.ready = 0;
	while (Mailbox.core.ready == 0)
		e_read(addr, (void *) (&Mailbox.core.ready), sz);

	return 0;
}


// Initialize result matrices
void matrix_init(int seed)
{
	int i, j, p;

	p = 0;
	for (i=0; i<_Sfft; i++)
		for (j=0; j<_Sfft; j++)
			Mailbox.B[p++] = 0x8dead;

	return;
}


// Generate Epiphany Core ID's
void init_coreID(unsigned int *coreID, int rows, int cols, unsigned int base_core)
{
	unsigned int cnum;
	unsigned int row, col;
	unsigned int base_row, base_col;

	base_row = base_core >> 6;
	base_col = base_core & 0x3f;

	cnum = 0;
	for (row=base_row; row<base_row+rows; row++)
		for (col=base_col; col<base_col+cols; col++)
		{
			coreID[cnum] = (row << 6) | (col << 0);
			cnum++;
		}

		return;
}
